import { API_BASE_URL } from "@/util/BaseUrl";
import { getCurrentUser } from "./AuthService";

const LISTING_API_URL = `${API_BASE_URL}/api/listing/`;

export const getSelfListing = async () => {
  const user = getCurrentUser();
  if (!user?.token) throw new Error("User not authenticated");

  const res = await fetch(`${LISTING_API_URL}self`, {
    headers: {
      "Content-Type": "application/json",
      Authorization: `Bearer ${user.token}`,
    },
    cache: "no-store",
  });

  const data = await res.json();
  if (!res.ok) {
    throw new Error(data.message || "Failed to get listing");
  }
  return data;
};

export const createListing = async (listingData) => {
  const user = getCurrentUser();
  if (!user?.token) throw new Error("User not authenticated");

  const res = await fetch(`${LISTING_API_URL}self/create`, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      Authorization: `Bearer ${user.token}`,
    },
    body: JSON.stringify(listingData),
  });

  const data = await res.json();
  if (!res.ok) {
    throw new Error(data.message || "Failed to create listing");
  }
  return data;
};

export const getMetaData = async () => {
  const res = await fetch(`${LISTING_API_URL}meta`, {next: { revalidate: 60 },});    
  const data = await res.json();
  if (!res.ok) {
    throw new Error(data.message || "Failed to fetch listing meta data");
  }
  return data;
};

// search listing
export const searchListing = async (searchRequest) => {
   // Delay function
  const delay = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

  // Wait 5 seconds before making the request
  await delay(2000);
  const res = await fetch(`${LISTING_API_URL}search`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(searchRequest),
  });

  if (res.ok) {
    const data = await res.json();
    return data;
  }
};

//get listing by id
export const getListingById = async (id) => {
  try {
    const res = await fetch(`${LISTING_API_URL}id/${id}`, {
      method: "GET",
      cache: "no-store", // Always get latest
      headers: {
        "Content-Type": "application/json",
      },
    });

    if (!res.ok) {
      const errorData = await res.json().catch(() => ({}));
      throw new Error(errorData.message || `Failed to fetch listing with ID ${id}`);
    }

    const data = await res.json();
    return data;
  } catch (error) {
    throw new Error(error.message || "Unknown error occurred while fetching listing");
  }
};

//upload images and video urls
export const uploadMedia = async (listingId, data) => {
  const user = getCurrentUser();
  if (!user?.token) throw new Error("User not authenticated");
  
  const formData = new FormData();

  // Append image files
  if (data.images && Array.isArray(data.images)) {
    data.images.forEach((file) => {
      formData.append("images", file);
    });
  }

  // Append video URLs
  if (data.videos && Array.isArray(data.videos)) {
    data.videos.forEach((url) => {
      formData.append("videos", url);
    });
  }

  const res = await fetch(`${LISTING_API_URL}self/upload-media/${listingId}`, {
    method: "POST",
    headers: {
      Authorization: `Bearer ${user.token}`,
    },
    body: formData,
  });
  
  if (!res.ok) {
    const data = await res.json();
    throw new Error(`Upload failed: ${data.message}`);
  }

};

//remove images and video urls
export const removeMedia = async (listingId, removeMedia) => {
  const user = getCurrentUser();
  if (!user?.token) throw new Error("User not authenticated");

  const res = await fetch(`${LISTING_API_URL}self/remove-media/${listingId}`, {
    method: "DELETE",
    headers: {
      "Content-Type": "application/json",
      Authorization: `Bearer ${user.token}`,
    },
    body: JSON.stringify(removeMedia),
  });

  if (!res.ok) {
    const data = await res.json();
    throw new Error(data.message);
  }
};