"use client";

import { useEffect, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import {
  Select, SelectTrigger, SelectContent, SelectItem, SelectValue
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Slider } from "@/components/ui/slider";
import { Accordion, AccordionItem, AccordionTrigger, AccordionContent } from "@/components/ui/accordion";
import { Car, Truck, Filter, RefreshCcw, MapPin } from "lucide-react";
import { getMetaData } from "@/service/ListingService";
import { getLocationListByName } from "@/service/LocationService";

export default function Filters({ setLoading }) {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [form, setForm] = useState({});
  const [meta, setMeta] = useState({});
  const [priceRange, setPriceRange] = useState([0, 200000]);

  useEffect(() => {
    getMetaData().then(setMeta);
  }, []);

  const handleChange = (name, value) => setForm(prev => ({ ...prev, [name]: value }));

  const handleClear = () => {
    setForm({});
    setPriceRange([0, 200000]);
    setLocationInput("");
    router.replace("/find-trucks?page=0");
  };

  useEffect(() => {
    const delay = setTimeout(() => {
      const params = new URLSearchParams();
      Object.entries(form).forEach(([key, val]) => val && params.set(key, val));
      params.set("page", 0);
      setLoading(true);
      router.replace(`/find-trucks?${params.toString()}`);
    }, 200);
    return () => clearTimeout(delay);
  }, [form]);
    
  const [locationInput, setLocationInput] = useState("");
  const [suggestions, setSuggestions] = useState([]);
  const [loadingSuggestions, setLoadingSuggestions] = useState(false);
  const [selectedLocation, setSelectedLocation] = useState("");

  useEffect(() => {
    const delay = setTimeout(async () => {
      if (locationInput.length >= 3 && locationInput !== selectedLocation) {
        try {
          setLoadingSuggestions(true);
          const results = await getLocationListByName(locationInput);
          setSuggestions(results || []);
        } catch (err) {
          console.error("Failed to fetch locations:", err);
        } finally {
          setLoadingSuggestions(false);
        }
      } else {
        setSuggestions([]);
      }
    }, 300); // smooth debounce

    return () => clearTimeout(delay);
  }, [locationInput]);

  const handleLocationSelect = (location) => {
    const { city, state, country, latitude, longitude } = location;

    setForm((prev) => ({
      ...prev,
      city: city?.name,
      state: state?.name,
      country: country?.name,
      lat: latitude,
      lng: longitude,
    }));

    setSelectedLocation(location.location);
    setLocationInput(location.location);
    setSuggestions([]);
  };

  const filteredModels = meta.models?.filter((m) => m.makerId === form.makerId);
  const filteredTrims = meta.trims?.filter((t) => t.modelId === form.modelId);
  const currentYear = new Date().getFullYear();
  const years = Array.from({ length: 20 }, (_, i) => currentYear - i);
  const transmissionSpeeds = Array.from({ length: 18 }, (_, i) => `${i + 1}-Speed`);
  
  return (
    <div className="bg-white p-4 space-y-3">
      <div className="flex items-center justify-between mb-2">
        <h2 className="text-lg font-semibold flex items-center gap-2 text-gray-700">
          <Filter className="w-5 h-5" /> Filters
        </h2>
        <Button variant="ghost" size="sm" onClick={handleClear} className="flex items-center gap-1">
          <RefreshCcw className="w-4 h-4" /> Clear
        </Button>
      </div>
      
      <Input placeholder="Keyword (Title/Desc)" onChange={e => handleChange("titleKeyword", e.target.value)} value={form.titleKeyword || ""}/>
      
      {/* Location Autocomplete */}
      <div className="relative">
        <label className="block text-sm font-medium text-gray-700 mb-1">
          <MapPin className="w-4 h-4 inline mr-1 text-gray-500" />
          Location
        </label>

        <div className="relative">
          <Input
            type="text"
            placeholder="Search city, state, or country..."
            value={locationInput}
            onChange={(e) => {
              const value = e.target.value;
              setLocationInput(value);
              setSelectedLocation("");
              if (value.trim() === "") {
                setForm((prev) => {
                  const updated = { ...prev };
                  delete updated.city;
                  delete updated.state;
                  delete updated.country;
                  delete updated.lat;
                  delete updated.lng;
                  return updated;
                });
              }
            }}
          />

          {/* Loading indicator */}
          {loadingSuggestions && (
            <div className="absolute top-full left-0 w-full bg-white text-gray-500 text-sm p-2 border border-t-0 border-gray-200 rounded-b-lg shadow-md z-20">
              Loading...
            </div>
          )}

          {/* Suggestions dropdown */}
          {suggestions.length > 0 && (
            <ul className="absolute top-full left-0 w-full bg-white border border-t-0 border-gray-200 rounded-b-lg shadow-md z-20 max-h-56 overflow-auto animate-in fade-in slide-in-from-top-1">
              {suggestions.map((loc) => (
                <li
                  key={loc.locationid}
                  className="px-3 py-2 cursor-pointer hover:bg-gray-100 flex items-center gap-2"
                  onClick={() => handleLocationSelect(loc)}
                >
                  <MapPin className="w-4 h-4 text-gray-400" />
                  <span className="text-gray-700">{loc.location}</span>
                </li>
              ))}
            </ul>
          )}
        </div>
      </div>

      {/* Price Range */}
      <div>
        <label className="text-sm text-gray-600">Price Range (${priceRange[0]} - ${priceRange[1]})</label>
        <Slider
          defaultValue={[0, 200000]}
          max={500000}
          step={5000}
          onValueChange={([min, max]) => {
            setPriceRange([min, max]);
            handleChange("minPrice", min);
            handleChange("maxPrice", max);
          }}
        />
      </div>

      {/* General */}
      <Accordion type="single" collapsible>
        <AccordionItem value="general">
          <AccordionTrigger>General Vehicle Info</AccordionTrigger>
          <AccordionContent className="space-y-2">

              <Select onValueChange={v => handleChange("year", v)} value={form.year || ""}>
                <SelectTrigger><SelectValue placeholder="Year From" /></SelectTrigger>
                <SelectContent>
                  {years.map(year => (
                    <SelectItem key={year} value={year.toString()}>{year}</SelectItem>
                  ))}
                </SelectContent>
              </Select>

              {/* Type + Condition */}
              <Select onValueChange={v => handleChange("type", v)} value={form.type || ""}>
                <SelectTrigger><SelectValue placeholder="Select Type" /></SelectTrigger>
                <SelectContent>
                  <SelectItem value="TRUCK"><Truck className="w-4 h-4 inline mr-2" />Truck</SelectItem>
                  <SelectItem value="TRAILER"><Car className="w-4 h-4 inline mr-2" />Trailer</SelectItem>
                </SelectContent>
              </Select>

              <Select onValueChange={v => handleChange("condition", v)} value={form.condition || ""}>
                <SelectTrigger><SelectValue placeholder="Condition" /></SelectTrigger>
                <SelectContent>
                  <SelectItem value="New">New</SelectItem>
                  <SelectItem value="Used">Used</SelectItem>
                </SelectContent>
              </Select>

              <Select onValueChange={v => handleChange("makerId", v)} value={form.makerId || ""}>
                <SelectTrigger><SelectValue placeholder="Makers" /></SelectTrigger>
                <SelectContent>
                  {meta.makers?.map(mk => (
                    <SelectItem key={mk.id} value={mk.id}>{mk.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select onValueChange={v => handleChange("modelId", v)} value={form.modelId || ""}>
                <SelectTrigger><SelectValue placeholder="Models" /></SelectTrigger>
                <SelectContent>
                  {filteredModels?.map(md => (
                    <SelectItem key={md.id} value={md.id}>{md.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select onValueChange={v => handleChange("trimId", v)} value={form.trimId || ""}>
                <SelectTrigger><SelectValue placeholder="Trims" /></SelectTrigger>
                <SelectContent>
                  {filteredTrims?.map(t => (
                    <SelectItem key={t.id} value={t.id}>{t.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
          </AccordionContent>
        </AccordionItem>

        {/* Truck */}
        <AccordionItem value="truck">
          <AccordionTrigger>Truck Filters</AccordionTrigger>
          <AccordionContent className="space-y-2">

            <Select onValueChange={v => handleChange("categoryId", v)} value={form.categoryId || ""}>
              <SelectTrigger>
                <SelectValue placeholder="Truck Category" />
              </SelectTrigger>
              <SelectContent>
                {meta.categories
                  ?.filter(t => t.type === "TRUCK")
                  .map(t => (
                    <SelectItem key={t.id} value={t.id}>
                      {t.name}
                    </SelectItem>
                  ))}
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("truckClassId", v)} value={form.truckClassId || ""}>
              <SelectTrigger><SelectValue placeholder="Truck Class" /></SelectTrigger>
              <SelectContent>
                {meta.truckClasses?.map(tc => (
                  <SelectItem key={tc.id} value={tc.id}>{`${tc.name} - ${tc.gvwrRange}`}</SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("fuelTypeId", v)} value={form.fuelTypeId || ""}>
              <SelectTrigger><SelectValue placeholder="Fuel Type" /></SelectTrigger>
              <SelectContent>
                {meta.fuelTypes?.map(ft => (
                  <SelectItem key={ft.id} value={ft.id}>{ft.name}</SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("transmission", v)} value={form.transmission || ""}>
              <SelectTrigger><SelectValue placeholder="Transmission" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="Automatic">Automatic</SelectItem>
                <SelectItem value="Manual">Manual</SelectItem>
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("transmissionSpeed", v)} value={form.transmissionSpeed || ""}>
              <SelectTrigger>
                <SelectValue placeholder="Transmission Speed" />
              </SelectTrigger>
              <SelectContent>
                {transmissionSpeeds.map(speed => (
                  <SelectItem key={speed} value={speed}>
                    {speed}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("driverSide", v)} value={form.driverSide || ""}>
              <SelectTrigger><SelectValue placeholder="Driver Side" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="Left">Left-hand</SelectItem>
                <SelectItem value="Right">Right-hand</SelectItem>
              </SelectContent>
            </Select>

            <Select onValueChange={v => handleChange("mileageRange", v)} value={form.mileageRange || ""}>
              <SelectTrigger>
                <SelectValue placeholder="Truck Mileage" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="upto 25000">Up to 25,000 km</SelectItem>
                <SelectItem value="upto 50000">Up to 50,000 km</SelectItem>
                <SelectItem value="upto 75000">Up to 75,000 km</SelectItem>
                <SelectItem value="upto 100000">Up to 100,000 km</SelectItem>
                <SelectItem value="upto 150000">Up to 150,000 km</SelectItem>
                <SelectItem value="upto 200000">Up to 200,000 km</SelectItem>
                <SelectItem value="above 200000">Above 200,000 km</SelectItem>
              </SelectContent>
            </Select>

            <Input placeholder="Engine Manufacturer" onChange={e => handleChange("engineManufacturer", e.target.value)} />

          </AccordionContent>
        </AccordionItem>

        {/* Trailer */}
        <AccordionItem value="trailer">
          <AccordionTrigger>Trailer Filters</AccordionTrigger>
          <AccordionContent className="space-y-2">

            <Select onValueChange={v => handleChange("categoryId", v)} value={form.categoryId || ""}>
              <SelectTrigger>
                <SelectValue placeholder="Trailer Category" />
              </SelectTrigger>
              <SelectContent>
                {meta.categories
                  ?.filter(t => t.type === "TRAILER")
                  .map(t => (
                    <SelectItem key={t.id} value={t.id}>
                      {t.name}
                    </SelectItem>
                  ))}
              </SelectContent>
            </Select>

            <div className="space-y-4">
              {/* Length Row */}
              <div className="flex flex-col sm:flex-row sm:space-x-4">
                <Input
                  placeholder="Min Length (m)"
                  value={form.trailerMinLength || ""}
                  onChange={e => handleChange("trailerMinLength", e.target.value)}
                  className="w-full sm:w-1/2"
                />
                <Input
                  placeholder="Max Length (m)"
                  value={form.trailerMaxLength || ""}
                  onChange={e => handleChange("trailerMaxLength", e.target.value)}
                  className="w-full sm:w-1/2 mt-2 sm:mt-0"
                />
              </div>

              {/* Width Row */}
              <div className="flex flex-col sm:flex-row sm:space-x-4">
                <Input
                  placeholder="Min Width (m)"
                  value={form.trailerMinWidth || ""}
                  onChange={e => handleChange("trailerMinWidth", e.target.value)}
                  className="w-full sm:w-1/2"
                />
                <Input
                  placeholder="Max Width (m)"
                  value={form.trailerMaxWidth || ""}
                  onChange={e => handleChange("trailerMaxWidth", e.target.value)}
                  className="w-full sm:w-1/2 mt-2 sm:mt-0"
                />
              </div>

              {/* Height Row */}
              <div className="flex flex-col sm:flex-row sm:space-x-4">
                <Input
                  placeholder="Min Height (m)"
                  value={form.trailerMinHeight || ""}
                  onChange={e => handleChange("trailerMinHeight", e.target.value)}
                  className="w-full sm:w-1/2"
                />
                <Input
                  placeholder="Max Height (m)"
                  value={form.trailerMaxHeight || ""}
                  onChange={e => handleChange("trailerMaxHeight", e.target.value)}
                  className="w-full sm:w-1/2 mt-2 sm:mt-0"
                />
              </div>
            </div>
            
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
            <Select onValueChange={v => handleChange("trailerComposition", v)}  value={form.trailerComposition || ""}>
              <SelectTrigger><SelectValue placeholder="Composition" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="Steel">Steel</SelectItem>
                <SelectItem value="Aluminum">Aluminum</SelectItem>
                <SelectItem value="Combo">Combo</SelectItem>
              </SelectContent>
            </Select>
            <Select onValueChange={v => handleChange("trailerFloorType", v)} value={form.trailerFloorType || ""}>
              <SelectTrigger><SelectValue placeholder="Floor Type" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="Wood">Wood</SelectItem>
                <SelectItem value="Steel">Steel</SelectItem>
                <SelectItem value="Aluminum">Aluminum</SelectItem>
              </SelectContent>
            </Select>
            <Select onValueChange={v => handleChange("trailerHitch", v)} value={form.trailerHitch || ""}>
              <SelectTrigger><SelectValue placeholder="Hitch Type" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="Gooseneck">Gooseneck</SelectItem>
                <SelectItem value="Fifth Wheel">Fifth Wheel</SelectItem>
                <SelectItem value="Bumper Pull">Bumper Pull</SelectItem>
              </SelectContent>
            </Select>
            <Select onValueChange={v => handleChange("trailerNumberOfAxles", v)} value={form.trailerNumberOfAxles || ""}>
              <SelectTrigger><SelectValue placeholder="Axles" /></SelectTrigger>
              <SelectContent>
                <SelectItem value="1">One</SelectItem>
                <SelectItem value="2">Two</SelectItem>
                <SelectItem value="3">Three</SelectItem>
              </SelectContent>
            </Select>
            </div>
          </AccordionContent>
        </AccordionItem>
      </Accordion>
    </div>
  );
}