"use client";

import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { getLocationListByName } from "@/service/LocationService";
import { Button } from "@/components/ui/button";
import { Search, Truck } from "lucide-react";
import Link from "next/link";

const Hero = () => {
  const router = useRouter();

  const [form, setForm] = useState({});
  const [locationInput, setLocationInput] = useState("");
  const [suggestions, setSuggestions] = useState([]);
  const [loadingSuggestions, setLoadingSuggestions] = useState(false);
  const [selectedLocation, setSelectedLocation] = useState("");

  // Debounced location search
  useEffect(() => {
    const delay = setTimeout(async () => {
      if (
        locationInput.length >= 3 &&
        locationInput !== selectedLocation
      ) {
        try {
          setLoadingSuggestions(true);
          const results = await getLocationListByName(locationInput);
          setSuggestions(results || []);
        } catch (err) {
          console.error("Failed to fetch locations:", err);
        } finally {
          setLoadingSuggestions(false);
        }
      } else {
        setSuggestions([]);
      }
    }, 300);
    return () => clearTimeout(delay);
  }, [locationInput]);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setForm((prev) => ({ ...prev, [name]: value }));
  };

  const handleLocationSelect = (location) => {
    const { city, state, country, latitude, longitude } = location;
    setForm((prev) => ({
      ...prev,
      city: city?.name,
      state: state?.name,
      country: country?.name,
      lat: latitude,
      lng: longitude,
    }));
    setSelectedLocation(location.location);
    setLocationInput(location.location);
    setSuggestions([]);
  };

  const handleSearch = () => {
    const params = new URLSearchParams();
    Object.entries(form).forEach(([key, value]) => {
      if (value) params.set(key, value);
    });
    params.set("page", 0);
    router.push(`/find-trucks?${params.toString()}`);
  };

  return (
    <section className="relative bg-gradient-to-r from-primary to-blue-700 text-white">
      <div className="absolute inset-0 bg-blue bg-opacity-40"></div>
      <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-24 lg:py-32">
        <div className="text-center">
          <h1 className="text-4xl lg:text-6xl font-bold mb-6">
            Find Your Perfect<br />
            <span className="text-chart-5">Commercial Truck</span>
          </h1>
          <p className="text-xl lg:text-2xl mb-10 max-w-3xl mx-auto">
            The most trusted marketplace for buying and selling commercial trucks, trailers, and heavy equipment
          </p>

          {/* CTA Buttons */}
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link href="/find-trucks">
              <Button size="lg" className="h-11 bg-chart-5 hover:bg-chart-5/90 text-lg !px-8 py-4">
                <Search className="mr-2 h-5 w-5" />
                Browse Inventory
              </Button>
            </Link>
            <Link href="/sell">
              <Button size="lg" variant="outline" className="h-11 border-2 border-white text-primary hover:bg-primary hover:text-white hover:border-primary text-lg !px-8 py-4">
                <Truck className="mr-2 h-5 w-5" />
                Sell Your Truck
              </Button>
            </Link>
          </div>

          {/* FILTERS INLINE */}
          <div className="flex flex-col lg:flex-row items-center justify-center gap-4 max-w-4xl mx-auto mt-10">
            {/* Type */}
            <select
              name="type"
              value={form.type || ""}
              onChange={handleChange}
              className="w-full lg:w-48 px-4 py-2 rounded-md bg-white text-black"
            >
              <option value="">All Types</option>
              <option value="TRUCK">Truck</option>
              <option value="TRAILER">Trailer</option>
            </select>

            {/* Location Autocomplete */}
            <div className="relative w-full lg:w-64">
              <input
                type="text"
                placeholder="City, State, Country"
                value={locationInput}
                onChange={(e) => {
                  const value = e.target.value;
                  setLocationInput(value);
                  setSelectedLocation("");
                  if (value.trim() === "") {
                    setForm((prev) => {
                      const updated = { ...prev };
                      delete updated.city;
                      delete updated.state;
                      delete updated.country;
                      delete updated.lat;
                      delete updated.lng;
                      return updated;
                    });
                  }
                }}
                className="w-full px-4 py-2 rounded-md bg-white text-black"
              />
              {loadingSuggestions && (
                <div className="absolute text-left left-0 top-full w-full bg-white text-sm text-gray-500 p-2 border border-t-0 border-gray-300 z-10">
                  Loading...
                </div>
              )}
              {suggestions.length > 0 && (
                <ul className="absolute text-left left-0 top-full w-full bg-white border border-t-0 border-gray-300 z-10 max-h-48 overflow-auto shadow-md text-black">
                  {suggestions.map((loc) => (
                    <li
                      key={loc.locationid}
                      className="px-4 py-2 hover:bg-gray-100 cursor-pointer"
                      onClick={() => handleLocationSelect(loc)}
                    >
                      {loc.location}
                    </li>
                  ))}
                </ul>
              )}
            </div>

            {/* Min Year */}
            <select
              name="year"
              value={form.year || ""}
              onChange={handleChange}
              className="w-full lg:w-40 px-4 py-2 rounded-md bg-white text-black"
            >
              <option value="">Min Year</option>
              {Array.from({ length: new Date().getFullYear() - 1999 }, (_, i) => 2000 + i)
                .reverse()
                .map((year) => (
                  <option key={year} value={year}>{year}</option>
                ))}
            </select>

            {/* Search Button */}
            <Button
              size="lg"
              onClick={handleSearch}
              className="h-11 bg-chart-5 hover:bg-chart-5/90 text-lg !px-6 py-4"
            >
              <Search className="mr-2 h-5 w-5" />
              Search
            </Button>
          </div>

        </div>
      </div>
    </section>
  );
};

export default Hero;
