'use client';

import { useContext, useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { Button } from '@/components/ui/button';
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion";
import Swal from 'sweetalert2';
import AuthContext from '@/app/context/AuthContext';
import { getMe } from '@/service/AuthService';
import { updateProfile } from '@/service/UserService';
import { CheckCircle, XCircle, PlusCircle } from 'lucide-react';
import Loader from '@/app/common/loader';
import Link from 'next/link';
import { renewRequest } from '@/service/UserPlanService';

export default function ProfilePage() {
  const { logout } = useContext(AuthContext);
  const router = useRouter();

  const [userData, setUserData] = useState(null);
  const [isEditing, setIsEditing] = useState(false);
  const [formData, setFormData] = useState({});

  // Fetch profile
  useEffect(() => {
    const fetchProfile = async () => {
      try {
        const data = await getMe();
        setUserData(data);
        setFormData({
          name: data.name || '',
          email: data.email || '',
          countryCode: data.countryCode || '',
          phone: data.phone || '',
          address: data.address || '',
          zipCode: data.zipCode || '',
          city: data.city || '',
          state: data.state || '',
          country: data.country || '',
        });
      } catch (error) {
        await logout();
        router.push('/signin');
      }
    };
    fetchProfile();
  }, []);

  const handleChange = (e) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
  };

  const handleSave = async () => {
    try {
      await updateProfile(formData);
      Swal.fire({
        icon: 'success',
        title: 'Profile updated!',
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 2000,
      });
      setUserData({ ...userData, ...formData });
      setIsEditing(false);
    } catch (error) {
      Swal.fire({
        icon: 'error',
        title: 'Update failed',
        text: error.message || 'Could not update profile',
      });
    }
  };

  const handleRenewRequest = async (userPlanId) => {
    try {
      const response = await renewRequest(userPlanId);

      Swal.fire({
        icon: "success",
        title: response.message || "Request Raised Successfully! Our team will contact you shortly.",
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: 2000,
        timerProgressBar: true, // ✅ adds a nice progress bar
      });

    } catch (error) {
      Swal.fire({
        icon: "error",
        title: "Request Failed",
        text: error.message || "Something went wrong. Please try again later.",
      });
    }
  };

  const countryCodes = [
    { code: "+1", name: "USA (+1)" },
    { code: "+44", name: "UK (+44)" },
    { code: "+91", name: "India (+91)" },
    { code: "+61", name: "Australia (+61)" },
    { code: "+81", name: "Japan (+81)" },
  ];

  const formatPrice = (price) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(price);
  };


  if (!userData) return <Loader message="Fetching your profile..." />;

  return (
    <main className="max-w-4xl mx-auto p-6 min-h-screen">
      <Accordion type="single" collapsible defaultValue="profile">
        {/* Profile Accordion */}
        <AccordionItem value="profile">
          <AccordionTrigger className="text-xl font-semibold">👤 My Profile</AccordionTrigger>
          <AccordionContent>
              {/* Profile Form */}
              <div className="p-4 space-y-6">
              <div className="space-y-6">
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block font-semibold mb-1">Name</label>
                    {isEditing ? (
                      <input name="name" value={formData.name} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.name}</p>
                    )}
                  </div>
                  <div>
                    <label className="block font-semibold mb-1">Email</label>
                    {isEditing ? (
                      <input type="email" name="email" value={formData.email} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.email}</p>
                    )}
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block font-semibold mb-1">Country Code</label>
                    {isEditing ? (
                      <select name="countryCode" value={formData.countryCode} onChange={handleChange} className="w-full border px-3 py-2 rounded-md">
                        {countryCodes.map((cc) => (
                          <option key={cc.code} value={cc.code}>{cc.name}</option>
                        ))}
                      </select>
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.countryCode}</p>
                    )}
                  </div>
                  <div>
                    <label className="block font-semibold mb-1">Phone</label>
                    {isEditing ? (
                      <input name="phone" value={formData.phone} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.phone}</p>
                    )}
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block font-semibold mb-1">Address</label>
                    {isEditing ? (
                      <input name="address" value={formData.address} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.address}</p>
                    )}
                  </div>
                  <div>
                    <label className="block font-semibold mb-1">Zip Code</label>
                    {isEditing ? (
                      <input name="zipCode" value={formData.zipCode} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.zipCode}</p>
                    )}
                  </div>
                </div>

                <div className="grid md:grid-cols-3 gap-4">
                  <div>
                    <label className="block font-semibold mb-1">City</label>
                    {isEditing ? (
                      <input name="city" value={formData.city} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.city}</p>
                    )}
                  </div>
                  <div>
                    <label className="block font-semibold mb-1">State</label>
                    {isEditing ? (
                      <input name="state" value={formData.state} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.state}</p>
                    )}
                  </div>
                  <div>
                    <label className="block font-semibold mb-1">Country</label>
                    {isEditing ? (
                      <input name="country" value={formData.country} onChange={handleChange} className="w-full border px-3 py-2 rounded-md" />
                    ) : (
                      <p className="border px-3 py-2 rounded-md bg-gray-100">{userData.country}</p>
                    )}
                  </div>
                </div>
              </div>

              <div className="mt-6 flex gap-4">
                {isEditing ? (
                  <>
                    <Button onClick={handleSave}>Save Changes</Button>
                    <Button variant="outline" onClick={() => setIsEditing(false)}>Cancel</Button>
                  </>
                ) : (
                  <Button onClick={() => setIsEditing(true)}>Edit Profile</Button>
                )}
              </div>
            </div>
          </AccordionContent>
        </AccordionItem>

        {/* User Plans Accordion */}
        <AccordionItem value="plans">
          <AccordionTrigger className="text-xl font-semibold">📦 My Plans</AccordionTrigger>
          <AccordionContent>
            <div className="grid md:grid-cols-2 gap-6 mt-4">
              {userData.userPlans?.length ? (
                <>
               {userData.userPlans.map((plan) => (
                  <div key={plan.id} className="p-5 border rounded-2xl shadow-md bg-white flex flex-col gap-2">
                    <div className="flex items-center justify-between">
                      <h2 className="text-2xl text-primary font-semibold">{plan.planName}</h2>
                      {plan.expired ? (
                        <span className="flex items-center text-red-600 text-sm"><XCircle className="mr-1 h-4 w-4" /> Expired</span>
                      ) : (
                        <span className="flex items-center text-green-600 text-sm"><CheckCircle className="mr-1 h-4 w-4" /> Active</span>
                      )}
                    </div>

                    <div>
                      <span className="text-xl font-semibold text-gray-900">
                        {plan.price === 0 ? "Free" : formatPrice(plan.price)}
                      </span>
                      {plan.price > 0 && (
                        <span className="text-gray-600">/Monthly</span>
                      )}
                    </div>

                    <ul className="text-sm space-y-1">
                      <li>Active Listings: {plan.activeListing}</li>
                      <li>Total Images: {plan.totalImage}</li>
                      <li>Total Videos: {plan.totalVideo}</li>
                      <li>Featured Listings: {plan.featuredListing}</li>
                      <li>Support: {plan.support}</li>
                      <li>Duration: {plan.duration}</li>
                    </ul>

                    <div className="text-xs text-gray-500 mt-2">
                      <p>Start: {new Date(plan.startDate).toLocaleDateString()}</p>
                      <p>End: {new Date(plan.endDate).toLocaleDateString()}</p>
                    </div>
                    {/* Renew Button only if expired */}
                    {plan.expired && (
                      <Button
                        onClick={() => handleRenewRequest(plan.id)}
                        variant="outline"
                        className="mt-3 bg-chart-5 hover:bg-chart-5/80"
                      >
                        Renew Request
                      </Button>
                    )}
                  </div>
                ))}
                
                {/* EXTRA Explore Plans Card */}
                <Link
                  href="/plans"
                  className="p-5 border rounded-2xl shadow-md bg-gray-50 hover:bg-gray-100 flex flex-col items-center justify-center text-center transition"
                >
                  <PlusCircle className="h-12 w-12 text-gray-400 mb-3" />
                  <span className="text-lg font-semibold text-gray-600">Explore Other Plans</span>
                </Link>
              </>
              ) : (
                // Show explore card even when no plans
                <Link
                  href="/plans"
                  className="p-5 border rounded-2xl shadow-md bg-gray-50 hover:bg-gray-100 flex flex-col items-center justify-center text-center transition"
                >
                  <PlusCircle className="h-12 w-12 text-gray-400 mb-3" />
                  <span className="text-lg font-semibold text-gray-600">Explore Plans</span>
                </Link>
              )}
            </div>
          </AccordionContent>
        </AccordionItem>
      </Accordion>
    </main>
  );
}
