"use client";

import { useEffect, useState, useContext } from "react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { getSelfListing, removeMedia, uploadMedia } from "@/service/ListingService";
import AuthContext from "@/app/context/AuthContext";
import Loader from "@/app/common/loader";
import { Anchor, CalendarDays, Cog, Compass, DollarSign, DoorOpen, Eye, Fuel, Gauge, Hash, ImageIcon, Layers, Layout, MapPin, Package, Palette, PlusCircle, Repeat, Ruler, Scale, Settings, Settings2, Tag, Truck, Wrench, X, Zap } from "lucide-react";
import Image from "next/image";
import { getMe } from "@/service/AuthService";
import { IMAGE_BASE_URL } from "@/util/BaseUrl";
import Swal from "sweetalert2";

export default function UserListingPage() {
  const [listings, setListings] = useState([]);
  const [userPlans, setUserPlans] = useState([]);
  const [loading, setLoading] = useState(true);
  const { logout } = useContext(AuthContext);
  const router = useRouter();
  const [previewListing, setPreviewListing] = useState(null);
  const [mediaListing, setMediaListing] = useState(null);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);

  const [selectedPlan, setSelectedPlan] = useState("all"); // filter state

  useEffect(() => {
    const fetchListings = async () => {
      try {
        const data = await getSelfListing();
        setListings(data);
      } catch (error) {
        await logout();
        router.push("/signin");
      } finally {
        setLoading(false);
      }
    };

    const fetchProfile = async () => {
      try {
        const data = await getMe();
        setUserPlans(data.userPlans || []);
      } catch (error) {
        console.error(error);
      }
    };

    fetchListings();
    fetchProfile();
  }, []);

  useEffect(() => {
    if (previewListing) {
      document.body.style.overflow = "hidden"; // prevent background scroll
    } else {
      document.body.style.overflow = ""; // reset
    }
    return () => {
      document.body.style.overflow = "";
    };
  }, [previewListing]);

  if (loading) return <Loader message="Fetching your listings..." />;

  // ✅ Apply filtering
  const filteredListings =
    selectedPlan === "all"
      ? listings
      : listings.filter((listing) => listing.plan?.id === selectedPlan);


  const handleUploadMedia = async (listingId, files, videoUrls = []) => {
    try {
      await uploadMedia(listingId, {
        images: files ? Array.from(files) : [],
        videos: videoUrls,

      });
      const data = await getSelfListing(); // refresh
      setListings(data);
      setMediaListing(data.find(l => l.id === listingId)); // update modal content too
    } catch (err) {
      Swal.fire({
        icon: "error",
        title: "Server Error",
        text: err.message || "Failed to Upload Media.",
      });
    }
  };

  const handleRemoveMedia = async (listingId, removePayload) => {
    try {
      await removeMedia(listingId, removePayload);
      const data = await getSelfListing(); // refresh
      setListings(data);
      setMediaListing(data.find(l => l.id === listingId));
    } catch (err) {
      Swal.fire({
        icon: "error",
        title: "Server Error",
        text: err.message || "Failed to Remove Media.",
      });
    }
  };


  return (
    <main className="min-h-screen p-4 bg-gray-50">
      <div className="max-w-6xl mx-auto">
        {listings.length > 0 ? (
          <div className="mb-6 flex items-center justify-between">
            <h1 className="text-3xl font-semibold text-primary">My Listings</h1>

            <div className="flex items-center gap-4">
              {/* Plan Filter */}
              {userPlans.length > 0 && (
                <div className="flex items-center gap-2">
                  <Tag className="w-5 h-5 text-gray-500" />
                  <select
                    value={selectedPlan}
                    onChange={(e) => setSelectedPlan(e.target.value)}
                    className="border border-gray-300 rounded-md px-4 py-2 bg-white shadow-sm focus:outline-none focus:ring-2 focus:ring-primary"
                  >
                    <option value="all">All Plans</option>
                    {userPlans.map((plan) => {
                      const count = listings.filter(l => l.plan?.id === plan.id).length;
                      return (
                        <option key={plan.id} value={plan.id}>
                          {plan.planName} ({count})
                        </option>
                      );
                    })}
                  </select>
                </div>
              )}

              <Link
                href="/sell"
                className="inline-flex items-center gap-2 border border-chart-5 text-chart-5 px-4 py-2 rounded-md hover:bg-chart-5 hover:text-white transition"
              >
                <PlusCircle className="w-5 h-5" />
                Sell Your Truck
              </Link>
            </div>
          </div>
        ) : (
          <h1 className="text-3xl font-semibold mb-6 text-primary">My Listings</h1>
        )}

        {listings.length === 0 ? (
          <div className="flex flex-col items-center justify-center p-8 bg-white rounded-lg shadow-md">
            <p className="text-lg mb-4">You don't have any listings yet.</p>
            <Link
              href="/sell"
              className="inline-flex items-center gap-2 bg-primary text-white px-5 py-3 rounded-md hover:bg-primary/90 transition"
            >
              <PlusCircle className="w-5 h-5" />
              Sell Your Truck
            </Link>
          </div>
        ) : (
          <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
            {filteredListings.map((listing) => (
              <div
                key={listing.id}
                className="bg-white rounded-lg shadow-md overflow-hidden flex flex-col"
              >

                <div className="relative w-full h-40">
                  {/* Manage Media Button */}
                  <button
                    onClick={() => setMediaListing(listing)}
                    className="absolute top-2 right-2 bg-white/80 p-2 rounded-full shadow hover:bg-white z-10"
                  >
                    <ImageIcon className="w-5 h-5 text-gray-700" />
                  </button>

                  {/* Show first image or fallback */}
                  {listing.images?.length > 0 ? (
                    <Image
                      src={`${IMAGE_BASE_URL}/${listing.images[0]}`}
                      alt={listing.title}
                      fill
                      className="object-cover"
                      sizes="100vw"
                    />
                  ) : (
                    <Image
                      src="/no-image.jpg"
                      alt="No Image"
                      fill
                      className="object-cover"
                      sizes="100vw"
                    />
                  )}
                </div>

                <div className="p-4 flex-1 flex flex-col">
                  {/* ✅ Add Plan info here */}
                  <div className="font-semibold text-chart-5">
                    {listing.plan?.planName || "N/A"}
                  </div>
                  <h2 className="text-lg font-bold mb-2">{listing.title}</h2>
                  <p className="text-gray-600 flex-1">{listing.description?.slice(0, 80)}{listing.description?.length > 80 && '...'}</p>
                  {/* Extra info */}
                  <div className="mt-4 grid grid-cols-2 gap-2 text-sm text-gray-700">
                    <div className="flex items-center gap-1">
                      <CalendarDays className="w-4 h-4" />
                      <span>{listing.year}</span>
                    </div>
                    {listing.type === "TRUCK" ? (
                      <div className="flex items-center gap-1">
                        <Gauge className="w-4 h-4" />
                        <span>{listing.truckMileage} mi</span>
                      </div>
                    ) : (
                      <div className="flex items-center gap-1">
                        <Settings className="w-4 h-4" />
                        <span>{listing.trailerType || "N/A"}</span>
                      </div>
                    )}
                    {listing.type === "TRUCK" ? (
                      <>
                        <div className="flex items-center gap-1">
                          <Zap className="w-4 h-4" />
                          <span>{listing.truckHorsePower || "N/A"} hp</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Fuel className="w-4 h-4" />
                          <span>{listing.truckFuelType || "N/A"}</span>
                        </div>
                      </>
                    ) : (
                      <>
                        <div className="flex items-center gap-1">
                          <Ruler className="w-4 h-4" />
                          <span>{listing.trailerLength || "N/A"} mtr</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <Gauge className="w-4 h-4" />
                          <span>{listing.trailerNumberOfAxles || "N/A"} axles</span>
                        </div>
                      </>
                    )}
                  </div>
                  <div className="mt-4 flex justify-between">
                    <p className="inline-flex items-center text-base font-bold text-primary">
                      <DollarSign className="w-5 h-5" />
                      <span className="text-lg">{listing.price}</span>
                    </p>
                    <button
                      onClick={() => setPreviewListing(listing)}
                      className="inline-flex items-center text-sm font-semibold bg-primary text-white px-4 py-2 rounded hover:bg-primary/90 transition"
                    >
                      <Eye className="w-4 h-4 mr-1" /> Preview
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {previewListing && (
        <div
          className="fixed inset-0 z-50 bg-black/60 flex items-center justify-center px-4"
          onClick={(e) => {
            if (e.target === e.currentTarget) setPreviewListing(null);
          }}
        >
          <div className="bg-white w-full max-w-4xl h-[90vh] rounded-xl shadow-xl overflow-hidden flex flex-col sm:flex-row relative">

            {/* Close Button */}
            <button
              onClick={() => setPreviewListing(null)}
              className="absolute top-4 right-4 z-50 text-gray-700 hover:text-red-500"
            >
              <X className="w-8 h-8" />
            </button>

            {/* Media Panel */}
            <div className="flex items-center justify-center w-full sm:w-[400px] bg-gray-100 p-4 sm:p-6">
              <div className="relative w-full max-w-[360px] aspect-[4/3] rounded-lg overflow-hidden shadow">
                {previewListing.images && previewListing.images.length > 0 ? (
                  <div className="relative w-full h-full">
                    <Image
                      src={`${IMAGE_BASE_URL}/${previewListing.images[currentImageIndex]}`}
                      alt={previewListing.title}
                      fill
                      className="object-cover"
                    />
                    {/* Prev / Next Buttons */}
                    <button
                      onClick={() =>
                        setCurrentImageIndex((prev) =>
                          prev === 0 ? previewListing.images.length - 1 : prev - 1
                        )
                      }
                      className="absolute left-2 top-1/2 -translate-y-1/2 bg-white/70 p-2 rounded-full shadow"
                    >
                      ◀
                    </button>
                    <button
                      onClick={() =>
                        setCurrentImageIndex((prev) =>
                          prev === previewListing.images.length - 1 ? 0 : prev + 1
                        )
                      }
                      className="absolute right-2 top-1/2 -translate-y-1/2 bg-white/70 p-2 rounded-full shadow"
                    >
                      ▶
                    </button>
                    {/* Image Counter */}
                    <div className="absolute bottom-2 right-2 bg-black text-white text-xs px-2 py-1 rounded-md">
                      {currentImageIndex + 1} / {previewListing.images.length}
                    </div>
                  </div>
                ) : (
                  <Image
                    src="/no-image.jpg"
                    alt="No Image"
                    fill
                    className="object-cover"
                  />
                )}
              </div>
            </div>

            {/* Divider */}
            <div className="hidden lg:block w-px bg-gray-200" />

            {/* Content Panel */}
            <div className="flex-1 p-6 overflow-y-auto">
              <h2 className="text-2xl font-bold mb-2 text-primary">{previewListing.title}</h2>

              <div className="border-t pt-4 mb-4 grid grid-cols-2 gap-x-6 gap-y-4 text-sm text-gray-800">
                <div className="flex items-center gap-2">
                  <CalendarDays className="w-5 h-5 text-gray-500" />
                  <span>{previewListing.year}</span>
                </div>
                <div className="flex items-center gap-2">
                  <MapPin className="w-5 h-5 text-gray-500" />
                  <span>{previewListing.city}, {previewListing.state}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Tag className="w-5 h-5 text-gray-500" />
                  <span>${previewListing.price}</span>
                </div>
                {previewListing.type === "TRUCK" ? (
                  <>
                    <div className="flex items-center gap-2">
                      <Gauge className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckMileage ? `${previewListing.truckMileage} mi` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Hash className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckVinNumber || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Zap className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckHorsePower ? `${previewListing.truckHorsePower} HP` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Package className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckStockNumber || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Cog className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckEngineManufacturer || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Wrench className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.engine || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Compass className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckDriverSide || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Truck className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckClassName || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Scale className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckClassGvwr || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Fuel className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.truckFuelType || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Repeat className="w-5 h-5 text-gray-500" />
                      <span>{`${previewListing.transmission} - ${previewListing.transmissionSpeed}` || "N/A"}</span>
                    </div>
                  </>
                ) : (
                  <>
                    <div className="flex items-center gap-2">
                      <Settings className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerType || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Ruler className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerLength ? `${previewListing.trailerLength} ft` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Hash className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerVinNumber || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Package className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerStockNumber || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Ruler className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerWidth ? `${previewListing.trailerWidth} ft` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Palette className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerColor || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Ruler className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerHeight ? `${previewListing.trailerHeight} ft` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Layers className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerComposition || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Gauge className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerNumberOfAxles ? `${previewListing.trailerNumberOfAxles} Axles` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Settings className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerAxleConfiguration || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Anchor className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerHitch || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Layout className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerFloorType || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <DoorOpen className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerDoorType || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <DoorOpen className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerNumberOfSideDoors ? `${previewListing.trailerNumberOfSideDoors} Side Doors` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <DoorOpen className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerRearOpening || "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Scale className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerWeightLbs ? `${previewListing.trailerWeightLbs} lbs` : "N/A"}</span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Scale className="w-5 h-5 text-gray-500" />
                      <span>{previewListing.trailerCapacityLbs ? `${previewListing.trailerCapacityLbs} lbs` : "N/A"}</span>
                    </div>
                  </>
                )}
              </div>
              <p className="border-t pt-2 text-gray-600 mb-4">{previewListing.description}</p>

              {/* Videos as links */}
              {previewListing.videos && previewListing.videos.length > 0 && (
                <div className="mt-4 space-y-2">
                  <h3 className="text-lg font-semibold text-primary">
                    Videos
                  </h3>
                  {previewListing.videos.map((url, idx) => (
                    <a
                      key={idx}
                      href={url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="block text-blue-600 hover:underline truncate"
                    >
                      {url}
                    </a>
                  ))}
                </div>
              )}

            </div>
          </div>
        </div>
      )}

      {/* Media Manager Modal */}
      {mediaListing && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg shadow-lg w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            {/* Header */}
            <div className="flex justify-between items-center p-4 border-b">
              <h2 className="text-lg font-semibold">
                Manage Media – {mediaListing.title}
              </h2>
              <button
                onClick={() => setMediaListing(null)}
                className="text-gray-500 hover:text-gray-800"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="p-4">
              {/* Images */}
              <h3 className="text-sm font-semibold mb-2">Images</h3>
              <div className="grid grid-cols-3 gap-2 mb-3">
                {mediaListing.images?.map((img) => (
                  <div key={img} className="relative group w-50 h-30">
                    <Image
                      src={`${IMAGE_BASE_URL}/${img}`}
                      alt="Listing image"
                      fill
                      className="rounded-md object-cover"
                    />
                    <button
                      onClick={() =>
                        handleRemoveMedia(mediaListing.id, { imageNamesToRemove: [img] })
                      }
                      className="absolute top-1 right-1 bg-red-600 text-white p-1 rounded-full"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  </div>
                ))}

                {/* Upload Button */}
                <label className="flex flex-col items-center justify-center border border-dashed rounded-md cursor-pointer hover:bg-gray-50 aspect-video">
                  <PlusCircle className="w-6 h-6 text-gray-500" />
                  <input
                    type="file"
                    accept="image/*"
                    multiple
                    className="hidden"
                    onChange={(e) =>
                      handleUploadMedia(mediaListing.id, e.target.files)
                    }
                  />
                </label>
              </div>

              {/* Videos */}
              <h3 className="text-sm font-semibold mb-2">Videos</h3>
              <div className="space-y-2">
                {mediaListing.videos?.map((url) => (
                  <div
                    key={url}
                    className="flex items-center justify-between text-sm border rounded px-2 py-1"
                  >
                    <a
                      href={url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="truncate text-blue-600 underline"
                    >
                      {url}
                    </a>
                    <button
                      onClick={() =>
                        handleRemoveMedia(mediaListing.id, { videoUrlsToRemove: [url] })
                      }
                      className="text-red-500 hover:text-red-700"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  </div>
                ))}

                {/* Add Video URL */}
                <form
                  onSubmit={(e) => {
                    e.preventDefault();
                    const url = e.currentTarget.video.value.trim();
                    if (url)
                      handleUploadMedia(mediaListing.id, null, [url]);
                    e.currentTarget.reset();
                  }}
                  className="flex items-center gap-2"
                >
                  <input
                    name="video"
                    type="url"
                    placeholder="Paste video URL"
                    className="flex-1 border rounded px-2 py-1 text-sm"
                  />
                  <button
                    type="submit"
                    className="px-3 py-1 bg-primary text-white rounded text-sm"
                  >
                    Add
                  </button>
                </form>
              </div>
            </div>
          </div>
        </div>
      )}

    </main>
  );
}
