"use client";

import { useState, useContext, useEffect } from "react";
import { useRouter } from "next/navigation";
import { changePassword } from "@/service/UserService";
import AuthContext from "@/app/context/AuthContext";
import { Eye, EyeOff } from "lucide-react";
import Swal from "sweetalert2";
import Loader from "@/app/common/loader"; // use your reusable Loader
import { getMe } from "@/service/AuthService";

export default function ChangePasswordPage() {
  const [oldPassword, setOldPassword] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [showOld, setShowOld] = useState(false);
  const [showNew, setShowNew] = useState(false);
  const [showConfirm, setShowConfirm] = useState(false);
  const [loading, setLoading] = useState(false);

  const { logout } = useContext(AuthContext);
  const router = useRouter();

    // Check auth & fetch profile
    useEffect(() => {
      const fetchProfile = async () => {
        try {
          await getMe();
        } catch (error) {
          console.error(error);
          await logout();
          router.push('/signin');
        }
      };
  
      fetchProfile();
    }, []);

  const validatePassword = (password) => {
    const regex = /^(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*]).{8,}$/;
    return regex.test(password);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();

    if (oldPassword === newPassword) {
      Swal.fire({
        icon: "error",
        title: "New password must be different from old password",
      });
      return;
    }

    if (!validatePassword(newPassword)) {
      Swal.fire({
        icon: "error",
        title: "Weak password",
        text: "New password must be at least 8 characters, with an uppercase letter, a number, and a special character.",
      });
      return;
    }

    if (newPassword !== confirmPassword) {
      Swal.fire({
        icon: "error",
        title: "Passwords do not match",
      });
      return;
    }

    setLoading(true);
    try {
      await changePassword({ oldPassword, newPassword });
      Swal.fire({
        icon: "success",
        title: "Password changed successfully",
        timer: 1500,
        showConfirmButton: false,
      });
      setOldPassword("");
      setNewPassword("");
      setConfirmPassword("");
      router.push("/user/profile");
    } catch (error) {
      console.error(error);
      Swal.fire({
        icon: "error",
        title: "Error",
        text: error.message || "Could not change password",
      });
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <Loader />;

  return (
    <main className="min-h-screen flex items-center justify-center bg-gray-50 p-4">
      <form
        onSubmit={handleSubmit}
        className="bg-white w-full max-w-md p-8 rounded-lg shadow-lg"
      >
        <h2 className="text-2xl font-bold mb-6 text-center">Change Password</h2>

        {/* Old Password */}
        <div className="mb-4">
          <label className="block mb-1 font-semibold">Old Password</label>
          <div className="relative">
            <input
              type={showOld ? "text" : "password"}
              value={oldPassword}
              onChange={(e) => setOldPassword(e.target.value)}
              required
              className="w-full px-4 py-2 border rounded-md pr-10 focus:outline-none focus:ring-2 focus:ring-primary"
            />
            <button
              type="button"
              onClick={() => setShowOld(!showOld)}
              className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500"
            >
              {showOld ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
            </button>
          </div>
        </div>

        {/* New Password */}
        <div className="mb-4">
          <label className="block mb-1 font-semibold">New Password</label>
          <div className="relative">
            <input
              type={showNew ? "text" : "password"}
              value={newPassword}
              onChange={(e) => setNewPassword(e.target.value)}
              required
              className="w-full px-4 py-2 border rounded-md pr-10 focus:outline-none focus:ring-2 focus:ring-primary"
            />
            <button
              type="button"
              onClick={() => setShowNew(!showNew)}
              className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500"
            >
              {showNew ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
            </button>
          </div>
          <p className="text-xs text-gray-500 mt-1">
            Must be at least 8 characters, with uppercase, number, and special character.
          </p>
        </div>

        {/* Confirm New Password */}
        <div className="mb-6">
          <label className="block mb-1 font-semibold">Confirm New Password</label>
          <div className="relative">
            <input
              type={showConfirm ? "text" : "password"}
              value={confirmPassword}
              onChange={(e) => setConfirmPassword(e.target.value)}
              required
              className="w-full px-4 py-2 border rounded-md pr-10 focus:outline-none focus:ring-2 focus:ring-primary"
            />
            <button
              type="button"
              onClick={() => setShowConfirm(!showConfirm)}
              className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500"
            >
              {showConfirm ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
            </button>
          </div>
        </div>

        <button
          type="submit"
          className="w-full bg-primary text-white py-2 rounded-md hover:bg-primary/90 transition"
        >
          Change Password
        </button>
      </form>
    </main>
  );
}
