"use client";

import { useContext, useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { register } from "@/service/UserService";
import Swal from "sweetalert2";
import { Eye, EyeOff } from "lucide-react";
import { sendOtp, verifyOtp } from "@/service/AuthService";
import AuthContext from "../context/AuthContext";

export default function RegisterPage() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    countryCode: "+1",
    phone: "",
    password: "",
    address: "",
    zipCode: "",
    city: "",
    state: "",
    country: "",
  });

  const [showPassword, setShowPassword] = useState(false);
  const [errors, setErrors] = useState({});
  const router = useRouter();
  const { login } = useContext(AuthContext);

  const handleChange = (e) => {
    setFormData((prev) => ({
      ...prev,
      [e.target.name]: e.target.value,
    }));
    setErrors((prev) => ({
      ...prev,
      [e.target.name]: "",
    }));
  };

  const validate = () => {
    const newErrors = {};

    if (!formData.name.trim()) newErrors.name = "Name is required";
    if (!formData.email.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/))
      newErrors.email = "Invalid email";
    if (!formData.phone.match(/^[0-9]{7,15}$/))
      newErrors.phone = "Phone must be 7-15 digits";

    // Strong password: same as backend rule
    const password = formData.password;
    const complexityPattern = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&/#^+=_()\[\]{}|:;\"'<>,.~`-]).{8,}$/;

    if (!password) {
      newErrors.password = "Password must not be empty.";
    } else if (password.length < 8) {
      newErrors.password = "Password must be at least 8 characters long.";
    } else if (!complexityPattern.test(password)) {
      newErrors.password =
        "Password must contain at least one uppercase letter, one lowercase letter, one digit, and one special character.";
    }
    if (!formData.address.trim()) newErrors.address = "Address is required";
    if (!formData.zipCode.trim()) newErrors.zipCode = "Zip Code is required";
    if (!formData.city.trim()) newErrors.city = "City is required";
    if (!formData.state.trim()) newErrors.state = "State is required";
    if (!formData.country.trim()) newErrors.country = "Country is required";

    setErrors(newErrors);

    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    if (!validate()) return;

    try {
      const phone = formData.countryCode + formData.phone;
      
      // Show loader while sending OTP
      Swal.fire({
        title: "Sending OTP...",
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();
        },
      });

      await sendOtp(phone);
      Swal.close(); // close loader

      // 2. Prompt user for OTP
      const { value: otp } = await Swal.fire({
        title: "Enter OTP",
        input: "text",
        inputLabel: `OTP sent to ${phone}`,
        inputPlaceholder: "Enter your OTP here",
        showCancelButton: true,
        confirmButtonText: "Verify",
        inputValidator: (value) => {
          if (!value) return "Please enter the OTP";
        },
      });
      
      if (!otp) return; // user cancelled

      // Loader while verifying OTP
      Swal.fire({
        title: "Verifying OTP...",
        allowOutsideClick: false,
        didOpen: () => {
          Swal.showLoading();
        },
      });

      const otpVerify = await verifyOtp(phone, otp);
      if (otpVerify.statusCode === 200) {
        // Loader while registering
        Swal.update({ title: "Registering..." });
        const res = await register(formData);

        // Loader while logging in
        Swal.update({ title: "Logging you in..." });
        await login(formData.email, formData.password);
        Swal.close();

        Swal.fire({
          icon: 'success',
          title: `${res.name} Registration successful!`,
          text: 'Welcome aboard!',
        }).then(() => {
          const pendingListing = localStorage.getItem("pendingListing");
          if (pendingListing) {
            router.push("/sell");
          } else {
            router.push("/user/listing");
          }
          // router.push("/user/listing");
        });        
      }
    } catch (error) {
      Swal.fire({
        icon: 'error',
        title: 'Registration failed',
        text: error.message || "Something went wrong!",
      });
    }
  };

  const countryCodes = [
    { code: "+1", name: "USA (+1)" },
    { code: "+44", name: "UK (+44)" },
    { code: "+91", name: "India (+91)" },
    { code: "+61", name: "Australia (+61)" },
    { code: "+81", name: "Japan (+81)" },
    // add more as needed
  ];

  return (
    <main className="min-h-screen flex items-center justify-center bg-gray-50 py-10">
      <form
        onSubmit={handleSubmit}
        className="bg-white p-8 rounded-lg shadow-md w-full max-w-3xl"
        noValidate
      >
        <h2 className="text-2xl font-bold mb-6 text-center">Register</h2>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {/* Name */}
          <div>
            <label className="block mb-1 font-semibold">Name</label>
            <input
              type="text"
              name="name"
              value={formData.name}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.name ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.name && (
              <p className="text-red-500 text-sm mt-1">{errors.name}</p>
            )}
          </div>

          {/* Email */}
          <div>
            <label className="block mb-1 font-semibold">Email</label>
            <input
              type="email"
              name="email"
              value={formData.email}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.email ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.email && (
              <p className="text-red-500 text-sm mt-1">{errors.email}</p>
            )}
          </div>

          {/* Country Code */}
          <div>
            <label className="block mb-1 font-semibold">Country Code</label>
            <select
              name="countryCode"
              value={formData.countryCode}
              onChange={handleChange}
              className="w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
            >
              {countryCodes.map((cc) => (
                <option key={cc.code} value={cc.code}>
                  {cc.name}
                </option>
              ))}
            </select>
          </div>

          {/* Phone */}
          <div>
            <label className="block mb-1 font-semibold">Phone</label>
            <input
              type="tel"
              name="phone"
              value={formData.phone}
              onChange={handleChange}
              required
              pattern="[0-9]{7,15}"
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.phone ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.phone && (
              <p className="text-red-500 text-sm mt-1">{errors.phone}</p>
            )}
          </div>

          {/* Password */}
          <div>
            <label className="block mb-1 font-semibold">Password</label>
            <div className="relative">
              <input
                type={showPassword ? "text" : "password"}
                name="password"
                value={formData.password}
                onChange={handleChange}
                required
                minLength={6}
                className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 pr-10 ${errors.password ? "border-red-500" : "focus:ring-blue-500"
                  }`}
              />
              <button
                type="button"
                onClick={() => setShowPassword(!showPassword)}
                className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 focus:outline-none"
              >
                {showPassword ? <EyeOff className="w-5 h-5" />: <Eye className="w-5 h-5"/>}
              </button>
            </div>
            <p className="text-gray-500 text-xs mt-1">
              Must be at least 8 characters, include uppercase, lowercase, number & special character.
            </p>
            {errors.password && (
              <p className="text-red-500 text-sm mt-1">{errors.password}</p>
            )}
          </div>


          {/* Address */}
          <div>
            <label className="block mb-1 font-semibold">Address</label>
            <input
              type="text"
              name="address"
              value={formData.address}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.address ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.address && (
              <p className="text-red-500 text-sm mt-1">{errors.address}</p>
            )}
          </div>

          {/* Zip Code */}
          <div>
            <label className="block mb-1 font-semibold">Zip Code</label>
            <input
              type="text"
              name="zipCode"
              value={formData.zipCode}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.zipCode ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.zipCode && (
              <p className="text-red-500 text-sm mt-1">{errors.zipCode}</p>
            )}
          </div>

          {/* City */}
          <div>
            <label className="block mb-1 font-semibold">City</label>
            <input
              type="text"
              name="city"
              value={formData.city}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.city ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.city && (
              <p className="text-red-500 text-sm mt-1">{errors.city}</p>
            )}
          </div>

          {/* State */}
          <div>
            <label className="block mb-1 font-semibold">State</label>
            <input
              type="text"
              name="state"
              value={formData.state}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.state ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.state && (
              <p className="text-red-500 text-sm mt-1">{errors.state}</p>
            )}
          </div>

          {/* Country */}
          <div>
            <label className="block mb-1 font-semibold">Country</label>
            <input
              type="text"
              name="country"
              value={formData.country}
              onChange={handleChange}
              required
              className={`w-full px-4 py-2 border rounded-md focus:outline-none focus:ring-2 ${errors.country ? "border-red-500" : "focus:ring-blue-500"
                }`}
            />
            {errors.country && (
              <p className="text-red-500 text-sm mt-1">{errors.country}</p>
            )}
          </div>
        </div>

        <button
          type="submit"
          className="w-full mt-6 bg-green-600 text-white py-2 rounded-md hover:bg-green-700 transition"
        >
          Register
        </button>

        <p className="mt-4 text-center text-sm text-gray-600">
          Already have an account?{" "}
          <Link href="/signin" className="text-blue-600 hover:underline">
            SignIn
          </Link>
        </p>
      </form>
    </main>
  );
}
