import ContactSalesModal from '@/components/forms/ContactSalesModal';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { getAllPlans } from '@/service/PlanService';
import { Building2, Calendar, Check, Crown, Gift, MessageSquare, Phone, Star, Truck, Upload, Users } from 'lucide-react';
import Link from 'next/link';
import React from 'react'

export default async function plans() {
    let plans = [];

    try {
        plans = await getAllPlans();
    } catch (error) {
        console.error("Error fetching plans:", error.message);
    }

    const formatPrice = (price) => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: "USD",
        }).format(price);
    };

    const getPlanDescription = (name) => {
        switch (name) {
            case "Basic":
                return "Get started with basic listing features at no cost";
            case "Starter":
                return "Perfect for individual sellers and small dealers";
            case "Business":
                return "Ideal for growing dealerships and fleet sellers";
            case "Premium":
                return "Complete solution for large dealerships and auction houses";
            default:
                return "";
        }
    };

    const getPlanSellingPoints = (plan) => {
        const points = [];

        if (plan.activeListing) {
            points.push(
                plan.activeListing === 999999
                    ? "Unlimited active listings"
                    : `${plan.activeListing} active listings`
            );
        }

        if (plan.featuredListing > 0) {
            points.push(`${plan.featuredListing} featured listings`);
        }

        if (plan.totalImage) {
            points.push(`${plan.totalImage} image uploads`);
        }

        if (plan.totalVideo) {
            points.push(`${plan.totalVideo} video uploads`);
        }

        if (plan.duration) {
            points.push(`Valid for ${plan.duration}`);
        }

        if (plan.support) {
            points.push(`${plan.support} support`);
        }

        if (plan.price === 0) {
            points.push("Free plan with no hidden charges");
        }

        return points;
    };


    return (
        <div className="min-h-screen bg-gray-50 py-12">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div className="text-center mb-12">
                    <h1 className="text-4xl font-bold text-gray-900 mb-4">
                        Choose Your Plan
                    </h1>
                    <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                        Select the perfect plan for your truck selling needs. Upgrade anytime.
                    </p>
                </div>

                {/* Plans Grid */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    {plans?.map((plan) => (
                        <Card
                            key={plan.id}
                            className={`relative transition-all duration-200 hover:shadow-lg ${plan.name === "Business"
                                    ? "border-chart-5 shadow-lg scale-105"
                                    : "border-gray-200"
                                }`}

                        >
                            {plan.name === "Business" && (
                                <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                                    <div className="bg-chart-5 text-white rounded-full px-4 py-1 flex items-center gap-2 shadow-md text-sm font-medium">
                                    <Star className="h-4 w-4" />
                                    <span>Most Popular</span>
                                    </div>
                                </div>
                            )}

                            <CardHeader className="text-center">
                                <div className="flex items-center justify-center mb-2">
                                    {plan.name === "Basic" && (
                                        <Gift className="h-6 w-6 text-green-600" />
                                    )}
                                    {plan.name === "Starter" && (
                                        <Users className="h-6 w-6 text-primary" />
                                    )}
                                    {plan.name === "Business" && (
                                        <Building2 className="h-6 w-6 text-chart-5" />
                                    )}
                                    {plan.name === "Premium" && (
                                        <Crown className="h-6 w-6 text-purple-600" />
                                    )}
                                </div>
                                <CardTitle className="text-2xl">{plan.name}</CardTitle>
                                <CardDescription className="text-sm">
                                    {getPlanDescription(plan.name)}
                                </CardDescription>
                                <div>
                                    <span className="text-4xl font-bold text-gray-900">
                                        {plan.price === 0 ? "Free" : formatPrice(plan.price)}
                                    </span>
                                    {plan.price > 0 && (
                                        <span className="text-gray-600">/Monthly</span>
                                    )}
                                </div>
                            </CardHeader>

                            <CardContent>
                                <div className="space-y-4">
                                    {/* Key Features */}
                                    <div className="flex items-center gap-2">
                                        <Truck className="h-4 w-4 text-primary" />
                                        <span className="text-sm">
                                            {plan.actveListing === 999999
                                                ? "Unlimited listings"
                                                : `Up to ${plan.activeListing} listings`}
                                        </span>
                                    </div>

                                    {(plan.featuredListing || 0) > 0 && (
                                        <div className="flex items-center gap-2">
                                            <Star className="h-4 w-4 text-chart-5" />
                                            <span className="text-sm">
                                                {plan.featuredListing} featured listings/month
                                            </span>
                                        </div>
                                    )}

                                    <div className="flex items-center gap-2">
                                        <Upload className="h-4 w-4 text-green-600" />
                                        <span className="text-sm">
                                            {plan.totalImage} images & {plan.totalVideo} videos
                                        </span>
                                    </div>

                                    <div className="flex items-center gap-2">
                                        <Calendar className="h-4 w-4 text-purple-600" />
                                        <span className="text-sm">{plan.duration}</span>
                                    </div>

                                    <div className="flex items-center gap-2">
                                        <Phone className="h-4 w-4 text-red-600" />
                                        <span className="text-sm">{plan.support} support</span>
                                    </div>

                                    <Separator />

                                    {/* All Features */}
                                    <div className="space-y-2">
                                        {getPlanSellingPoints(plan).map((point, index) => (
                                            <div key={index} className="flex items-center gap-2">
                                                <Check className="h-3 w-3 text-green-500 flex-shrink-0" />
                                                <span className="text-xs text-gray-600">{point}</span>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            </CardContent>

                            <CardFooter>
                                {plan.name === "Basic" ? (
                                    <Link href="/sell" className="w-full">
                                        <Button className="w-full">Start Free</Button>
                                    </Link>
                                ) : (
                                    <ContactSalesModal plan={plan} />
                                )}
                            </CardFooter>
                        </Card>
                    ))}
                </div>

                {/* Additional Info */}
                <div className="mt-12 text-center">
                    <Card className="max-w-2xl mx-auto">
                        <CardHeader>
                            <CardTitle className="text-2xl">Need Help Choosing?</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-4 text-left">
                                <div>
                                    <p className="font-semibold text-sm">Starter Plan</p>
                                    <p className="text-xs text-gray-600">
                                        Perfect for individual sellers with a few trucks to sell
                                        occasionally.
                                    </p>
                                </div>
                                <div>
                                    <p className="font-semibold text-sm">Business Plan</p>
                                    <p className="text-xs text-gray-600">
                                        Ideal for small to medium dealers who need enhanced
                                        visibility and video support.
                                    </p>
                                </div>
                                <div>
                                    <p className="font-semibold text-sm">Premium Plan</p>
                                    <p className="text-xs text-gray-600">
                                        Complete solution for large dealers and auction houses with
                                        high-volume needs.
                                    </p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Manual Process Information */}
                <div className="mt-12">
                    <Card className="max-w-4xl mx-auto border-blue-200 bg-blue-50">
                        <CardHeader>
                            <CardTitle className="text-2xl flex items-center gap-2 text-primary">
                                <MessageSquare className="h-5 w-5" />
                                How Our Sales Process Works
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid md:grid-cols-3 gap-6 text-center">
                                <div className="space-y-2">
                                    <div className="w-10 h-10 bg-primary text-white rounded-full flex items-center justify-center mx-auto font-semibold">
                                        1
                                    </div>
                                    <h4 className="font-semibold text-gray-900">
                                        Submit Request
                                    </h4>
                                    <p className="text-sm text-gray-600">
                                        Fill out the contact form with your requirements and
                                        business details
                                    </p>
                                </div>
                                <div className="space-y-2">
                                    <div className="w-10 h-10 bg-chart-5 text-white rounded-full flex items-center justify-center mx-auto font-semibold">
                                        2
                                    </div>
                                    <h4 className="font-semibold text-gray-900">
                                        Sales Consultation
                                    </h4>
                                    <p className="text-sm text-gray-600">
                                        Our team contacts you within 24 hours for a personalized
                                        consultation
                                    </p>
                                </div>
                                <div className="space-y-2">
                                    <div className="w-10 h-10 bg-green-600 text-white rounded-full flex items-center justify-center mx-auto font-semibold">
                                        3
                                    </div>
                                    <h4 className="font-semibold text-gray-900">
                                        Manual Activation
                                    </h4>
                                    <p className="text-sm text-gray-600">
                                        We activate your plan manually after payment is processed by
                                        our team
                                    </p>
                                </div>
                            </div>

                            <div className="mt-6 p-4 bg-white rounded-lg border border-blue-200">
                                <h4 className="font-semibold text-gray-900 mb-2">
                                    Why Manual Processing?
                                </h4>
                                <ul className="text-sm text-gray-600 space-y-1">
                                    <li>
                                        • Personalized plan customization based on your specific
                                        needs
                                    </li>
                                    <li>
                                        • Custom payment terms and invoice options for businesses
                                    </li>
                                    <li>
                                        • Direct support channel setup and account configuration
                                    </li>
                                    <li>
                                        • Volume discounts and enterprise feature negotiations
                                    </li>
                                </ul>
                            </div>
                        </CardContent>
                    </Card>
                </div>                
            </div>
        </div>
    )
}